/*
 * Decompiled with CFR 0.152.
 */
package guideme.internal.search;

import guideme.Guide;
import guideme.Guides;
import guideme.compiler.FrontmatterNavigation;
import guideme.compiler.IndexingSink;
import guideme.compiler.ParsedGuidePage;
import guideme.document.DefaultStyles;
import guideme.document.flow.LytFlowContent;
import guideme.document.flow.LytFlowSpan;
import guideme.internal.search.Analyzers;
import guideme.internal.search.GuideQueryParser;
import guideme.internal.search.IndexSchema;
import guideme.internal.search.LanguageSpecificAnalyzerWrapper;
import guideme.internal.search.PageIndexer;
import guideme.internal.shaded.lucene.analysis.Analyzer;
import guideme.internal.shaded.lucene.document.Document;
import guideme.internal.shaded.lucene.document.Field;
import guideme.internal.shaded.lucene.document.StoredField;
import guideme.internal.shaded.lucene.document.StringField;
import guideme.internal.shaded.lucene.document.TextField;
import guideme.internal.shaded.lucene.index.DirectoryReader;
import guideme.internal.shaded.lucene.index.IndexReader;
import guideme.internal.shaded.lucene.index.IndexWriter;
import guideme.internal.shaded.lucene.index.IndexWriterConfig;
import guideme.internal.shaded.lucene.index.StoredFields;
import guideme.internal.shaded.lucene.index.Term;
import guideme.internal.shaded.lucene.search.BooleanClause;
import guideme.internal.shaded.lucene.search.BooleanQuery;
import guideme.internal.shaded.lucene.search.IndexSearcher;
import guideme.internal.shaded.lucene.search.PhraseQuery;
import guideme.internal.shaded.lucene.search.Query;
import guideme.internal.shaded.lucene.search.ScoreDoc;
import guideme.internal.shaded.lucene.search.TermQuery;
import guideme.internal.shaded.lucene.search.TopDocs;
import guideme.internal.shaded.lucene.search.highlight.Highlighter;
import guideme.internal.shaded.lucene.search.highlight.InvalidTokenOffsetsException;
import guideme.internal.shaded.lucene.search.highlight.QueryScorer;
import guideme.internal.shaded.lucene.store.ByteBuffersDirectory;
import guideme.internal.util.LangUtil;
import guideme.libs.mdast.model.MdAstAnyContent;
import guideme.libs.mdast.model.MdAstHeading;
import guideme.libs.unist.UnistNode;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import net.minecraft.resources.ResourceLocation;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GuideSearch
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(GuideSearch.class);
    private static final long TIME_PER_TICK = TimeUnit.MILLISECONDS.toNanos(5L);
    private final ByteBuffersDirectory directory = new ByteBuffersDirectory();
    private final Analyzer analyzer;
    private final IndexWriter indexWriter;
    private IndexReader indexReader;
    private final List<GuideIndexingTask> pendingTasks = new ArrayList<GuideIndexingTask>();
    private Instant indexingStarted;
    private int pagesIndexed;
    private final Set<String> warnedAboutLanguage = Collections.synchronizedSet(new HashSet());
    private final Set<String> indexedLanguages = Collections.synchronizedSet(new HashSet());

    public GuideSearch() {
        this.analyzer = new LanguageSpecificAnalyzerWrapper();
        IndexWriterConfig config = new IndexWriterConfig(this.analyzer);
        try {
            this.indexWriter = new IndexWriter(this.directory, config);
            this.indexWriter.flush();
            this.indexWriter.commit();
            this.indexReader = DirectoryReader.open(this.directory);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Failed to create index writer.", e);
        }
    }

    public void index(Guide guide) {
        try {
            this.indexWriter.deleteDocuments(new PhraseQuery("guide_id", guide.getId().toString()));
        }
        catch (IOException e) {
            LOG.error("Failed to delete all documents before re-indexing.", (Throwable)e);
        }
        if (this.pendingTasks.isEmpty()) {
            this.indexingStarted = Instant.now();
            this.pagesIndexed = 0;
        }
        this.pendingTasks.removeIf(t -> t.guide.getId().equals((Object)guide.getId()));
        this.pendingTasks.add(new GuideIndexingTask(guide, new ArrayList<ParsedGuidePage>(guide.getPages())));
    }

    public void indexAll() {
        for (Guide guide : Guides.getAll()) {
            this.index(guide);
        }
    }

    public void processWork() {
        if (this.pendingTasks.isEmpty()) {
            return;
        }
        long start = System.nanoTime();
        Iterator<GuideIndexingTask> guideTaskIt = this.pendingTasks.iterator();
        while (guideTaskIt.hasNext()) {
            if (this.isTimeElapsed(start)) {
                return;
            }
            GuideIndexingTask guideTask = guideTaskIt.next();
            Guide guide = guideTask.guide();
            Iterator<ParsedGuidePage> pageIt = guideTask.pendingPages.iterator();
            while (pageIt.hasNext()) {
                if (this.isTimeElapsed(start)) {
                    return;
                }
                ParsedGuidePage page = pageIt.next();
                Document pageDoc = this.createPageDocument(guideTask.guide(), page);
                if (pageDoc != null) {
                    try {
                        this.indexWriter.addDocument(pageDoc);
                    }
                    catch (IOException e) {
                        LOG.error("Failed to index document {}{}", new Object[]{guide, page, e});
                    }
                    String searchLang = pageDoc.get("search_lang");
                    if (searchLang != null) {
                        this.indexedLanguages.add(searchLang);
                    }
                }
                ++this.pagesIndexed;
                pageIt.remove();
            }
            guideTaskIt.remove();
        }
        try {
            this.indexWriter.flush();
            this.indexWriter.commit();
            this.indexReader.close();
            this.indexReader = DirectoryReader.open(this.directory);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        LOG.info("Indexing of {} pages finished in {}", (Object)this.pagesIndexed, (Object)Duration.between(this.indexingStarted, Instant.now()));
    }

    private boolean isTimeElapsed(long start) {
        return System.nanoTime() - start >= TIME_PER_TICK;
    }

    public List<SearchResult> searchGuide(String queryText, @Nullable Guide onlyFromGuide) {
        TopDocs topDocs;
        Query query;
        if (queryText.isEmpty()) {
            return List.of();
        }
        String searchLanguage = this.getLuceneLanguageFromMinecraft(LangUtil.getCurrentLanguage());
        IndexSearcher indexSearcher = new IndexSearcher(this.indexReader);
        try {
            query = GuideQueryParser.parse(queryText, this.analyzer, this.indexedLanguages);
        }
        catch (Exception e) {
            LOG.debug("Failed to parse search query: '{}'", (Object)queryText, (Object)e);
            return List.of();
        }
        if (onlyFromGuide != null) {
            query = new BooleanQuery.Builder().add(query, BooleanClause.Occur.MUST).add(new TermQuery(new Term("guide_id", onlyFromGuide.getId().toString())), BooleanClause.Occur.FILTER).build();
        }
        LOG.debug("Running GuideME search query: {}", (Object)query);
        try {
            topDocs = indexSearcher.search(query, 25);
        }
        catch (IOException e) {
            LOG.error("Failed to search for '{}'", (Object)queryText, (Object)e);
            return List.of();
        }
        ArrayList<SearchResult> result = new ArrayList<SearchResult>();
        Highlighter highlighter = new Highlighter(new QueryScorer(query));
        try {
            StoredFields storedFields = this.indexReader.storedFields();
            for (ScoreDoc scoreDoc : topDocs.scoreDocs) {
                Document document = storedFields.document(scoreDoc.doc);
                ResourceLocation guideId = new ResourceLocation(document.get("guide_id"));
                ResourceLocation pageId = new ResourceLocation(document.get("page_id"));
                Guide guide = Guides.getById(guideId);
                if (guide == null) {
                    LOG.warn("Search index produced guide id {} which couldn't be found.", (Object)guideId);
                    continue;
                }
                ParsedGuidePage page = guide.getParsedPage(pageId);
                if (page == null) {
                    LOG.warn("Search index produced page {} in guide {}, which couldn't be found.", (Object)pageId, (Object)guideId);
                    continue;
                }
                String bestFragment = "";
                try {
                    bestFragment = highlighter.getBestFragment(this.analyzer, IndexSchema.getTextField(searchLanguage), document.get("page_content"));
                    if (bestFragment == null) {
                        bestFragment = "";
                    }
                }
                catch (InvalidTokenOffsetsException e) {
                    LOG.error("Cannot determine text to highlight for result", (Throwable)e);
                }
                String pageTitle = document.get("page_title");
                int startOfSegment = 0;
                LytFlowSpan currentSpan = new LytFlowSpan();
                for (int i = 0; i < bestFragment.length(); ++i) {
                    if (this.isStartOfHighlight(bestFragment, i)) {
                        currentSpan.appendText(bestFragment.substring(startOfSegment, i));
                        startOfSegment = i + 3;
                        LytFlowSpan parentSpan = currentSpan;
                        currentSpan = new LytFlowSpan();
                        currentSpan.setStyle(DefaultStyles.SEARCH_RESULT_HIGHLIGHT);
                        parentSpan.append(currentSpan);
                        continue;
                    }
                    if (!this.isEndOfHighlight(bestFragment, i)) continue;
                    currentSpan.appendText(bestFragment.substring(startOfSegment, i));
                    startOfSegment = i + 4;
                    currentSpan = Objects.requireNonNull((LytFlowSpan)currentSpan.getFlowParent());
                }
                currentSpan.appendText(bestFragment.substring(startOfSegment));
                result.add(new SearchResult(guideId, pageId, pageTitle, currentSpan));
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return result;
    }

    private boolean isStartOfHighlight(CharSequence text, int i) {
        return i + 3 <= text.length() && text.charAt(i) == '<' && text.charAt(i + 1) == 'B' && text.charAt(i + 2) == '>';
    }

    private boolean isEndOfHighlight(CharSequence text, int i) {
        return i + 4 <= text.length() && text.charAt(i) == '<' && text.charAt(i + 1) == '/' && text.charAt(i + 2) == 'B' && text.charAt(i + 3) == '>';
    }

    @Nullable
    private Document createPageDocument(Guide guide, ParsedGuidePage page) {
        String pageText = GuideSearch.getSearchableText(guide, page);
        String pageTitle = GuideSearch.getPageTitle(guide, page);
        String searchLang = this.getLuceneLanguageFromMinecraft(page.getLanguage());
        Document doc = new Document();
        doc.add(new StringField("guide_id", guide.getId().toString(), Field.Store.YES));
        doc.add(new StoredField("page_id", page.getId().toString()));
        doc.add(new StoredField("lang", page.getLanguage()));
        doc.add(new StoredField("search_lang", searchLang));
        doc.add(new StoredField("page_title", pageTitle));
        doc.add(new StoredField("page_content", pageText));
        doc.add(new TextField(IndexSchema.getTitleField(searchLang), pageTitle, Field.Store.NO));
        doc.add(new TextField(IndexSchema.getTextField(searchLang), pageText, Field.Store.NO));
        return doc;
    }

    private String getLuceneLanguageFromMinecraft(String language) {
        String luceneLang = Analyzers.MINECRAFT_TO_LUCENE_LANG.get(language);
        if (luceneLang == null) {
            if (this.warnedAboutLanguage.add(language)) {
                LOG.warn("Minecraft language '{}' has unknown and will be treated as english for the purposes of search.", (Object)language);
            }
            return "en";
        }
        return luceneLang;
    }

    private static String getPageTitle(Guide guide, ParsedGuidePage page) {
        FrontmatterNavigation navigationEntry = page.getFrontmatter().navigationEntry();
        if (navigationEntry != null) {
            return navigationEntry.title();
        }
        for (MdAstAnyContent child : page.getAstRoot().children()) {
            if (!(child instanceof MdAstHeading)) continue;
            MdAstHeading heading = (MdAstHeading)child;
            if (heading.depth != 1) continue;
            final StringBuilder pageTitle = new StringBuilder();
            IndexingSink sink = new IndexingSink(){

                @Override
                public void appendText(UnistNode parent, String text) {
                    pageTitle.append(text);
                }

                @Override
                public void appendBreak() {
                    pageTitle.append(' ');
                }
            };
            new PageIndexer(guide, guide.getExtensions(), page.getId()).indexContent(heading.children(), sink);
            return pageTitle.toString();
        }
        return page.getId().toString();
    }

    private static String getSearchableText(Guide guide, ParsedGuidePage page) {
        final StringBuilder searchableText = new StringBuilder();
        IndexingSink sink = new IndexingSink(){

            @Override
            public void appendText(UnistNode parent, String text) {
                searchableText.append(text);
            }

            @Override
            public void appendBreak() {
                searchableText.append('\n');
            }
        };
        new PageIndexer(guide, guide.getExtensions(), page.getId()).index(page.getAstRoot(), sink);
        return searchableText.toString();
    }

    @Override
    public void close() throws IOException {
        IOException suppressed = null;
        try {
            this.indexWriter.close();
        }
        catch (IOException e) {
            suppressed = e;
        }
        try {
            this.indexReader.close();
        }
        catch (IOException e) {
            if (suppressed != null) {
                suppressed.addSuppressed(e);
            }
            suppressed = e;
        }
        try {
            this.directory.close();
        }
        catch (IOException e) {
            if (suppressed != null) {
                e.addSuppressed(suppressed);
            }
            throw e;
        }
        if (suppressed != null) {
            throw suppressed;
        }
    }

    record GuideIndexingTask(Guide guide, List<ParsedGuidePage> pendingPages) {
    }

    public record SearchResult(ResourceLocation guideId, ResourceLocation pageId, String pageTitle, LytFlowContent text) {
        public SearchResult {
            Objects.requireNonNull(guideId, "guideId");
            Objects.requireNonNull(pageId, "pageId");
            Objects.requireNonNull(pageTitle, "pageTitle");
            Objects.requireNonNull(text, "text");
        }
    }
}

